// ============================================================================
// ============================================================================
// ============================================================================
// ==                                                                        ==
// == Name    : TheEmuLib.Emu_TaShe_XY.fsh                                   ==
// == Type    : Fragment shader                                              ==
// == Version : 1.0.3 (2017/01/29)                                           ==
// == Creator : TheEmu © TheEmu 2017, Some Rights Reserved                   ==
// == Licence : Creative Commons Attribution-ShareAlike 4.0                  ==
// ==           http://creativecommons.org/licences/by-sa/4.0                ==
// ==                                                                        ==
// == Purpose: To apply a combined 2D taper and shear transform to an image. ==
// ==                                                                        ==
// == Description: The source image is distorted using the taper transform   ==
// ==                                                                        ==
// ==        x' = x * (1-a)*y                                                ==
// ==        y' = y * (1-b)*x                                                ==
// ==                                                                        ==
// == and the shear transform                                                ==
// ==                                                                        ==
// ==        x' = x + a*y                                                    ==
// ==        y' = y + b*x                                                    ==
// ==                                                                        ==
// == where x and y are the normalised coordinates ranging from 0.0 to 1.0.  ==
// ==                                                                        ==
// == Shear transforms rectangles into parallelograms.                       ==
// ==                                                                        ==
// == Taper transforms a rectangle into a trapezium if applied along an axis ==
// == parallel to one of the sides of the rectangle, to a kite when tapering ==
// == is symetric or to an irregular quadrilateral if there is no symetry.   ==
// ==                                                                        ==
// == Optionaly X and Y scale factors may be specified causing the image  to ==
// == be resized accordingly.  The transform may be increase the size of the ==
// == image and scale factors of less than 1.0 than one may be needed if the ==
// == transformed image is to fit into the graphics window.                  ==
// ==                                                                        ==
// == The hotspot for the transform specifies a point acting as its  centre, ==
// == i.e. the origin of the (x,y) coordinates.  At  this  point x and y are ==
// == both zero so the point is unaffected by the transform.  The default is ==
// == is (0.0,0.0), i.e. at a corner of the source image. Using a hotspot of ==
// == (0.5,0.5) will keep the image centered,  unfortunately this can not be ==
// == made the default as (0.0,0.0) is a valid hotspot.  Making (0,0) be the ==
// == centre of the image would have been possible,  but that would not have ==
// == been compatible with the normal iStripper conventions.                 ==
// ==                                                                        ==
// == This shader is a combination of the Shear_XY and Taper_XY shaders.  It ==
// == can be used in place of either but is slightly less efficient and does ==
// == not have a self-documenting name so if you want a pure shear or a pure ==
// == taper transform then using Shear_XY or Taper_XY is preferable.         ==
// ==                                                                        ==
// == ====================================================================== ==
// ==                                                                        ==
// == This file is a member of The Emu's shader library.                     ==
// ==                                                                        ==
// == ====================================================================== ==
// ==                                                                        ==
// == Update history:                                                        ==
// ==                                                                        ==
// ==   2017/01/09 - v1.0.0 - Initial version.                               ==
// ==   2017/01/11 - v1.0.1 - Minor tidy up and improved comments.           ==
// ==   2017/01/13 - v1.0.2 - Corrected a comment.                           ==
// ==   2017/01/29 - v1.0.2 - Updated comments.                              ==
// ==                                                                        ==
// ============================================================================
// ============================================================================
// ============================================================================

// ============================================================================
// == Standard shader inputs ==================================================
// ============================================================================

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// The image that is to be manipulated.

uniform sampler2D iChannel0;

// ============================================================================
// == Imports from TheEmuLib ==================================================
// ============================================================================
//
// The GLSL shader language currently provides no mechanism for importing  any
// elements that are defined in other modules, not even C's crude source level
// #include mechanism. In the absence of anything better TheEmuLib handles any
// imports by manualy copying relevent utility code snippets from the  sources
// in the Shader Lib.Inc directory. This is very crude but I have attempted to
// be systematic in the way in which this is presented in the library sources.
//
// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Common_Utilities.lib.src

#define EMU_DEFAULT(type,x,default_value) ( (x==type(0.0)) ? (default_value) : (x) )

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Functions from TheEmuLib.Emu_Coordinate_Normalisation.lib.src

#define EMU_NORMALISE_TO_WINDOW_1(xy,wsize) ( (xy)/(wsize) )

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

vec2 Emu_Normalise_to_Window ( vec2 xy )
 { return EMU_NORMALISE_TO_WINDOW_1 ( xy, u_WindowSize.xy );
 }

// ============================================================================
// == Shader specific inputs ==================================================
// ============================================================================

// EmuLib standard scale and hotspot parameters. Note, a hotspot of (0.5,0.5)
// is required to produce a symetric result.

uniform vec2 Emu_Shear_XY_scale;
uniform vec2 Emu_Shear_XY_hotspot;

vec2 scale   = EMU_DEFAULT ( vec2, Emu_TaShe_XY_scale,   vec2(1.0) );
vec2 hotspot = EMU_DEFAULT ( vec2, Emu_TaShe_XY_hotspot, vec2(0.0) );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// The shear and taper for X and Y are specified by the following parameters.
// Value of zero for an axis produces no distortion.  Values  are expected to
// be in the range -1.0 to +1.0, but values outside that range are valid.

uniform vec2 Emu_TaShe_XY_shear;
uniform vec2 Emu_TaShe_XY_taper;

#define shear Emu_TaShe_XY_shear
#define taper Emu_TaShe_XY_taper

// ============================================================================
// == The shader's major function =============================================
// ============================================================================

vec4 Emu_TaShe_XY ( vec2 uv )
 {
   // Apply the transform with its fixed point at the hotspot.  Note, the code
   // here is the inverse of the required transform because we are determining
   // what point in the source image will be transformed to the current pixel.

   vec2 st = ( uv - hotspot ) / scale;

   st = st - shear*st.yx;
   st = st / ( 1.0 - taper*st.yx );

   st = st + hotspot;

   // Determine the colour using transparent black if the point
   // lies outside of the area of the transformed source image.

   return ( st == fract(abs(st)) ) ? texture2D(iChannel0,st) : vec4(0.0);

 }

// ============================================================================
// == The shader's main routine ===============================================
// ============================================================================

void main ( void )
 {
   // Get the normalised coordinates of the current point.

   vec2 uv = Emu_Normalise_to_Window ( gl_FragCoord.xy );

   // Apply the transform and update the shader's outputs.

   gl_FragColor = Emu_TaShe_XY(uv) * gl_Color;

 }

// ============================================================================
